<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Movie Display</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            background-color: #222;
            font-family: Arial, sans-serif;
            overflow: hidden;
            color: white;
            width: 100vw;
            height: 100vh;
        }

        .backdrop-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            z-index: -1;
        }

        .backdrop {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .backdrop-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, rgba(0,0,0,0.9) 20%, rgba(0,0,0,0.6) 50%, rgba(0,0,0,0.4) 100%);
        }

        .main-wrapper {
            position: fixed;
            width: 100vw;
            height: 100vh;
            display: flex;
        }

        .left-panel {
            width: 25vw;
            height: 100vh;
            padding: 2vh 2vw;
            background: rgba(0,0,0,0.5);
            display: flex;
            flex-direction: column;
        }

        .right-panel {
            width: 75vw;
            height: 100vh;
            padding: 2vh;
            display: flex;
            flex-direction: column;
            gap: 2vh;
        }

        .logo-section {
            height: 15vh;
            display: flex;
            justify-content: center;
            align-items: center;
            margin-bottom: 2vh;
        }

        .logo {
            max-height: 100%;
            max-width: 90%;
            object-fit: contain;
        }

        .rating-container {
            text-align: center;
            height: 8vh;
        }

        .rating {
            font-size: 3vh;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 1vw;
        }

        .star {
            color: #ffd700;
        }

        .info-grid {
            display: grid;
            gap: 1.5vh;
            margin-top: 2vh;
            background: rgba(0,0,0,0.3);
            padding: 1.5vh;
            border-radius: 1vh;
        }

        .info-item {
            display: flex;
            align-items: center;
            gap: 1vw;
            font-size: 1.8vh;
            height: 3vh;
        }

        .icon {
            width: 2vw;
            text-align: center;
            font-size: 2vh;
        }

        .images-section {
            height: 40vh;
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 1vw;
        }

        .movie-image {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 1vh;
            cursor: pointer;
            transition: transform 0.3s ease;
        }

        .movie-image:hover {
            transform: scale(1.02);
        }

        .synopsis-container {
            height: 30vh;
            background: rgba(0,0,0,0.5);
            padding: 2vh;
            border-radius: 1vh;
            overflow: hidden;
        }

        .synopsis {
            font-size: 2vh;
            line-height: 1.6;
            height: 100%;
            overflow-y: auto;
        }

        .genres {
            display: flex;
            flex-wrap: wrap;
            gap: 1vh;
            margin: 2vh 0;
            justify-content: center;
        }

        .genre-tag {
            background: rgba(255,255,255,0.1);
            padding: 0.8vh 1.5vh;
            border-radius: 2vh;
            font-size: 1.5vh;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            background: rgba(0,0,0,0.9);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            max-width: 90vw;
            max-height: 90vh;
            object-fit: contain;
        }

        .modal-close {
            position: absolute;
            top: 2vh;
            right: 2vw;
            color: white;
            font-size: 3vh;
            cursor: pointer;
            background: none;
            border: none;
        }

        ::-webkit-scrollbar {
            width: 0.5vw;
            height: 0.5vw;
        }

        ::-webkit-scrollbar-track {
            background: rgba(255,255,255,0.1);
        }

        ::-webkit-scrollbar-thumb {
            background: rgba(255,255,255,0.3);
            border-radius: 1vh;
        }
    </style>
</head>
<body>
    <div class="backdrop-container">
        <img id="backdrop" class="backdrop">
        <div class="backdrop-overlay"></div>
    </div>
    
    <div class="main-wrapper">
        <div class="left-panel">
            <div class="logo-section">
                <img id="logo" class="logo">
            </div>
            
            <div class="rating-container">
                <div class="rating">
                    <span class="star">★</span>
                    <span id="rating">0.0/10</span>
                </div>
                <div id="trending"></div>
            </div>

            <div class="genres" id="genres"></div>
            
            <div class="info-grid">
                <div class="info-item">
                    <span class="icon">⏱</span>
                    <span id="runtime"></span>
                </div>
                <div class="info-item">
                    <span class="icon">📅</span>
                    <span id="releaseYear"></span>
                </div>
                <div class="info-item">
                    <span class="icon">🌐</span>
                    <span id="language"></span>
                </div>
                <div class="info-item">
                    <span class="icon">👥</span>
                    <span id="voteCount"></span>
                </div>
                <div class="info-item">
                    <span class="icon">💰</span>
                    <span id="budget"></span>
                </div>
                <div class="info-item">
                    <span class="icon">💎</span>
                    <span id="revenue"></span>
                </div>
            </div>
        </div>

        <div class="right-panel">
            <div class="images-section" id="imagesGrid"></div>
            <div class="synopsis-container">
                <div class="synopsis" id="synopsis"></div>
            </div>
        </div>
    </div>

    <div id="imageModal" class="modal">
        <span class="modal-close">&times;</span>
        <img class="modal-content" id="modalImage">
    </div>
</body>
</html>
<script>
        const apiKey = '6b8e3eaa1a03ebb45642e9531d8a76d2';
        let currentIndex = 0;
        let movieIds = [];

        async function fetchMovieImages(movieId) {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/movie/${movieId}/images?api_key=${apiKey}`);
                if (!response.ok) throw new Error('Failed to fetch movie images');
                const data = await response.json();
                // Filtrar backdrops sin idioma específico para evitar logos promocionales
                const backdrops = data.backdrops.filter(img => !img.iso_639_1);
                // Obtener 3 imágenes aleatorias
                return shuffleArray(backdrops).slice(0, 3);
            } catch (error) {
                console.error('Error fetching movie images:', error);
                return [];
            }
        }

        function shuffleArray(array) {
            const newArray = [...array];
            for (let i = newArray.length - 1; i > 0; i--) {
                const j = Math.floor(Math.random() * (i + 1));
                [newArray[i], newArray[j]] = [newArray[j], newArray[i]];
            }
            return newArray;
        }

        function updateImages(images) {
            const gallery = document.getElementById('imagesGrid');
            gallery.innerHTML = '';

            images.forEach(image => {
                const img = document.createElement('img');
                img.className = 'movie-image';
                img.src = `https://image.tmdb.org/t/p/w780${image.file_path}`;
                img.onclick = () => {
                    const modal = document.getElementById('imageModal');
                    const modalImg = document.getElementById('modalImage');
                    modal.style.display = 'flex';
                    modalImg.src = `https://image.tmdb.org/t/p/original${image.file_path}`;
                };
                gallery.appendChild(img);
            });
        }

        // Modal handlers
        document.querySelector('.modal-close').onclick = function() {
            document.getElementById('imageModal').style.display = 'none';
        }

        window.onclick = function(event) {
            const modal = document.getElementById('imageModal');
            if (event.target === modal) {
                modal.style.display = 'none';
            }
        }

        async function fetchMovieDetails(movieId) {
            try {
                const details = await fetch(`https://api.themoviedb.org/3/movie/${movieId}?api_key=${apiKey}&append_to_response=images`)
                    .then(r => r.json());
                return { details };
            } catch (error) {
                console.error('Error fetching movie details:', error);
                return null;
            }
        }

        function formatMoney(amount) {
            return new Intl.NumberFormat('en-US', {
                style: 'currency',
                currency: 'USD',
                notation: 'compact',
                maximumFractionDigits: 1
            }).format(amount);
        }

        function formatNumber(number) {
            return new Intl.NumberFormat().format(number);
        }

        async function updateMovieDisplay() {
            if (movieIds.length === 0) return;

            const movieId = movieIds[currentIndex];
            try {
                const [movieData, images] = await Promise.all([
                    fetchMovieDetails(movieId),
                    fetchMovieImages(movieId)
                ]);

                if (!movieData) return;
                const { details } = movieData;

                // Update backdrop
                document.getElementById('backdrop').src = 
                    `https://image.tmdb.org/t/p/original${details.backdrop_path}`;

                // Update logo
                const logo = details.images.logos?.find(logo => logo.iso_639_1 === 'en');
                const logoElement = document.getElementById('logo');
                if (logo) {
                    logoElement.src = `https://image.tmdb.org/t/p/w500${logo.file_path}`;
                    logoElement.style.display = 'block';
                } else {
                    logoElement.style.display = 'none';
                }

                // Update text content
                document.getElementById('synopsis').textContent = details.overview;
                document.getElementById('rating').textContent = 
                    `${details.vote_average.toFixed(1)}/10`;
                document.getElementById('trending').textContent = 
                    `#${currentIndex + 1} Trending`;

                // Update genres
                const genresContainer = document.getElementById('genres');
                genresContainer.innerHTML = details.genres
                    .map(genre => `<span class="genre-tag">${genre.name}</span>`)
                    .join('');

                // Update movie info
                document.getElementById('runtime').textContent = 
                    `${details.runtime} min`;
                document.getElementById('releaseYear').textContent = 
                    new Date(details.release_date).getFullYear();
                document.getElementById('language').textContent = 
                    details.original_language.toUpperCase();
                document.getElementById('voteCount').textContent = 
                    `${formatNumber(details.vote_count)} votos`;
                document.getElementById('budget').textContent = 
                    formatMoney(details.budget);
                document.getElementById('revenue').textContent = 
                    formatMoney(details.revenue);

                // Update images
                updateImages(images);

                currentIndex = (currentIndex + 1) % movieIds.length;
            } catch (error) {
                console.error('Error updating movie display:', error);
            }
        }

        async function initialize() {
            try {
                const response = await fetch(
                    `https://api.themoviedb.org/3/discover/movie?api_key=${apiKey}&sort_by=popularity.desc&language=en`
                );
                const data = await response.json();
                movieIds = data.results.map(movie => movie.id);
                await updateMovieDisplay();
            } catch (error) {
                console.error('Error initializing:', error);
            }
        }

        // Initialize the display
        initialize();

        // Update every 8 seconds
        setInterval(updateMovieDisplay, 8000);
    </script>
</body>
</html>